"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.type = exports.serializedType = void 0;
exports.assert = assert;
exports.deserialize = deserialize;
exports.from = from;
exports.getSignPayload = getSignPayload;
exports.hash = hash;
exports.serialize = serialize;
exports.toRpc = toRpc;
exports.validate = validate;
const AccessList = require("./AccessList.js");
const Address = require("./Address.js");
const Hash = require("./Hash.js");
const Hex = require("./Hex.js");
const Rlp = require("./Rlp.js");
const Signature = require("./Signature.js");
const TransactionEnvelope = require("./TransactionEnvelope.js");
exports.serializedType = '0x01';
exports.type = 'eip2930';
function assert(envelope) {
    const { chainId, gasPrice, to } = envelope;
    if (chainId <= 0)
        throw new TransactionEnvelope.InvalidChainIdError({ chainId });
    if (to)
        Address.assert(to, { strict: false });
    if (gasPrice && BigInt(gasPrice) > 2n ** 256n - 1n)
        throw new TransactionEnvelope.GasPriceTooHighError({ gasPrice });
}
function deserialize(serialized) {
    const transactionArray = Rlp.toHex(Hex.slice(serialized, 1));
    const [chainId, nonce, gasPrice, gas, to, value, data, accessList, yParity, r, s,] = transactionArray;
    if (!(transactionArray.length === 8 || transactionArray.length === 11))
        throw new TransactionEnvelope.InvalidSerializedError({
            attributes: {
                chainId,
                nonce,
                gasPrice,
                gas,
                to,
                value,
                data,
                accessList,
                ...(transactionArray.length > 8
                    ? {
                        yParity,
                        r,
                        s,
                    }
                    : {}),
            },
            serialized,
            type: exports.type,
        });
    let transaction = {
        chainId: Number(chainId),
        type: exports.type,
    };
    if (Hex.validate(to) && to !== '0x')
        transaction.to = to;
    if (Hex.validate(gas) && gas !== '0x')
        transaction.gas = BigInt(gas);
    if (Hex.validate(data) && data !== '0x')
        transaction.data = data;
    if (Hex.validate(nonce))
        transaction.nonce = nonce === '0x' ? 0n : BigInt(nonce);
    if (Hex.validate(value) && value !== '0x')
        transaction.value = BigInt(value);
    if (Hex.validate(gasPrice) && gasPrice !== '0x')
        transaction.gasPrice = BigInt(gasPrice);
    if (accessList.length !== 0 && accessList !== '0x')
        transaction.accessList = AccessList.fromTupleList(accessList);
    const signature = r && s && yParity ? Signature.fromTuple([yParity, r, s]) : undefined;
    if (signature)
        transaction = {
            ...transaction,
            ...signature,
        };
    assert(transaction);
    return transaction;
}
function from(envelope, options = {}) {
    const { signature } = options;
    const envelope_ = (typeof envelope === 'string' ? deserialize(envelope) : envelope);
    assert(envelope_);
    return {
        ...envelope_,
        ...(signature ? Signature.from(signature) : {}),
        type: 'eip2930',
    };
}
function getSignPayload(envelope) {
    return hash(envelope, { presign: true });
}
function hash(envelope, options = {}) {
    const { presign } = options;
    return Hash.keccak256(serialize({
        ...envelope,
        ...(presign
            ? {
                r: undefined,
                s: undefined,
                yParity: undefined,
                v: undefined,
            }
            : {}),
    }));
}
function serialize(envelope, options = {}) {
    const { chainId, gas, data, input, nonce, to, value, accessList, gasPrice } = envelope;
    assert(envelope);
    const accessTupleList = AccessList.toTupleList(accessList);
    const signature = Signature.extract(options.signature || envelope);
    const serialized = [
        Hex.fromNumber(chainId),
        nonce ? Hex.fromNumber(nonce) : '0x',
        gasPrice ? Hex.fromNumber(gasPrice) : '0x',
        gas ? Hex.fromNumber(gas) : '0x',
        to ?? '0x',
        value ? Hex.fromNumber(value) : '0x',
        data ?? input ?? '0x',
        accessTupleList,
        ...(signature ? Signature.toTuple(signature) : []),
    ];
    return Hex.concat('0x01', Rlp.fromHex(serialized));
}
function toRpc(envelope) {
    const signature = Signature.extract(envelope);
    return {
        ...envelope,
        chainId: Hex.fromNumber(envelope.chainId),
        data: envelope.data ?? envelope.input,
        ...(typeof envelope.gas === 'bigint'
            ? { gas: Hex.fromNumber(envelope.gas) }
            : {}),
        ...(typeof envelope.nonce === 'bigint'
            ? { nonce: Hex.fromNumber(envelope.nonce) }
            : {}),
        ...(typeof envelope.value === 'bigint'
            ? { value: Hex.fromNumber(envelope.value) }
            : {}),
        ...(typeof envelope.gasPrice === 'bigint'
            ? { gasPrice: Hex.fromNumber(envelope.gasPrice) }
            : {}),
        type: '0x1',
        ...(signature ? Signature.toRpc(signature) : {}),
    };
}
function validate(envelope) {
    try {
        assert(envelope);
        return true;
    }
    catch {
        return false;
    }
}
//# sourceMappingURL=TransactionEnvelopeEip2930.js.map