"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.type = exports.serializedType = void 0;
exports.assert = assert;
exports.deserialize = deserialize;
exports.from = from;
exports.getSignPayload = getSignPayload;
exports.hash = hash;
exports.serialize = serialize;
exports.toRpc = toRpc;
exports.validate = validate;
const AccessList = require("./AccessList.js");
const Blobs = require("./Blobs.js");
const Hash = require("./Hash.js");
const Hex = require("./Hex.js");
const Kzg = require("./Kzg.js");
const Rlp = require("./Rlp.js");
const Signature = require("./Signature.js");
const TransactionEnvelope = require("./TransactionEnvelope.js");
const TransactionEnvelopeEip1559 = require("./TransactionEnvelopeEip1559.js");
exports.serializedType = '0x03';
exports.type = 'eip4844';
function assert(envelope) {
    const { blobVersionedHashes } = envelope;
    if (blobVersionedHashes) {
        if (blobVersionedHashes.length === 0)
            throw new Blobs.EmptyBlobVersionedHashesError();
        for (const hash of blobVersionedHashes) {
            const size = Hex.size(hash);
            const version = Hex.toNumber(Hex.slice(hash, 0, 1));
            if (size !== 32)
                throw new Blobs.InvalidVersionedHashSizeError({ hash, size });
            if (version !== Kzg.versionedHashVersion)
                throw new Blobs.InvalidVersionedHashVersionError({
                    hash,
                    version,
                });
        }
    }
    TransactionEnvelopeEip1559.assert(envelope);
}
function deserialize(serialized) {
    const transactionOrWrapperArray = Rlp.toHex(Hex.slice(serialized, 1));
    const hasNetworkWrapper = transactionOrWrapperArray.length === 4;
    const transactionArray = hasNetworkWrapper
        ? transactionOrWrapperArray[0]
        : transactionOrWrapperArray;
    const wrapperArray = hasNetworkWrapper
        ? transactionOrWrapperArray.slice(1)
        : [];
    const [chainId, nonce, maxPriorityFeePerGas, maxFeePerGas, gas, to, value, data, accessList, maxFeePerBlobGas, blobVersionedHashes, yParity, r, s,] = transactionArray;
    const [blobs, commitments, proofs] = wrapperArray;
    if (!(transactionArray.length === 11 || transactionArray.length === 14))
        throw new TransactionEnvelope.InvalidSerializedError({
            attributes: {
                chainId,
                nonce,
                maxPriorityFeePerGas,
                maxFeePerGas,
                gas,
                to,
                value,
                data,
                accessList,
                ...(transactionArray.length > 9
                    ? {
                        yParity,
                        r,
                        s,
                    }
                    : {}),
            },
            serialized,
            type: exports.type,
        });
    let transaction = {
        blobVersionedHashes: blobVersionedHashes,
        chainId: Number(chainId),
        type: exports.type,
    };
    if (Hex.validate(to) && to !== '0x')
        transaction.to = to;
    if (Hex.validate(gas) && gas !== '0x')
        transaction.gas = BigInt(gas);
    if (Hex.validate(data) && data !== '0x')
        transaction.data = data;
    if (Hex.validate(nonce))
        transaction.nonce = nonce === '0x' ? 0n : BigInt(nonce);
    if (Hex.validate(value) && value !== '0x')
        transaction.value = BigInt(value);
    if (Hex.validate(maxFeePerBlobGas) && maxFeePerBlobGas !== '0x')
        transaction.maxFeePerBlobGas = BigInt(maxFeePerBlobGas);
    if (Hex.validate(maxFeePerGas) && maxFeePerGas !== '0x')
        transaction.maxFeePerGas = BigInt(maxFeePerGas);
    if (Hex.validate(maxPriorityFeePerGas) && maxPriorityFeePerGas !== '0x')
        transaction.maxPriorityFeePerGas = BigInt(maxPriorityFeePerGas);
    if (accessList?.length !== 0 && accessList !== '0x')
        transaction.accessList = AccessList.fromTupleList(accessList);
    if (blobs && commitments && proofs)
        transaction.sidecars = Blobs.toSidecars(blobs, {
            commitments: commitments,
            proofs: proofs,
        });
    const signature = r && s && yParity
        ? Signature.fromTuple([yParity, r, s])
        : undefined;
    if (signature)
        transaction = {
            ...transaction,
            ...signature,
        };
    assert(transaction);
    return transaction;
}
function from(envelope, options = {}) {
    const { signature } = options;
    const envelope_ = (typeof envelope === 'string' ? deserialize(envelope) : envelope);
    assert(envelope_);
    return {
        ...envelope_,
        ...(signature ? Signature.from(signature) : {}),
        type: 'eip4844',
    };
}
function getSignPayload(envelope) {
    return hash(envelope, { presign: true });
}
function hash(envelope, options = {}) {
    const { presign } = options;
    return Hash.keccak256(serialize({
        ...envelope,
        ...(presign
            ? {
                sidecars: undefined,
                r: undefined,
                s: undefined,
                yParity: undefined,
                v: undefined,
            }
            : {}),
    }));
}
function serialize(envelope, options = {}) {
    const { blobVersionedHashes, chainId, gas, nonce, to, value, maxFeePerBlobGas, maxFeePerGas, maxPriorityFeePerGas, accessList, data, } = envelope;
    assert(envelope);
    const accessTupleList = AccessList.toTupleList(accessList);
    const signature = Signature.extract(options.signature || envelope);
    const serialized = [
        Hex.fromNumber(chainId),
        nonce ? Hex.fromNumber(nonce) : '0x',
        maxPriorityFeePerGas ? Hex.fromNumber(maxPriorityFeePerGas) : '0x',
        maxFeePerGas ? Hex.fromNumber(maxFeePerGas) : '0x',
        gas ? Hex.fromNumber(gas) : '0x',
        to ?? '0x',
        value ? Hex.fromNumber(value) : '0x',
        data ?? '0x',
        accessTupleList,
        maxFeePerBlobGas ? Hex.fromNumber(maxFeePerBlobGas) : '0x',
        blobVersionedHashes ?? [],
        ...(signature ? Signature.toTuple(signature) : []),
    ];
    const sidecars = options.sidecars || envelope.sidecars;
    const blobs = [];
    const commitments = [];
    const proofs = [];
    if (sidecars)
        for (let i = 0; i < sidecars.length; i++) {
            const { blob, commitment, proof } = sidecars[i];
            blobs.push(blob);
            commitments.push(commitment);
            proofs.push(proof);
        }
    return Hex.concat('0x03', sidecars
        ?
            Rlp.fromHex([serialized, blobs, commitments, proofs])
        :
            Rlp.fromHex(serialized));
}
function toRpc(envelope) {
    const signature = Signature.extract(envelope);
    return {
        ...envelope,
        chainId: Hex.fromNumber(envelope.chainId),
        data: envelope.data ?? envelope.input,
        ...(typeof envelope.gas === 'bigint'
            ? { gas: Hex.fromNumber(envelope.gas) }
            : {}),
        ...(typeof envelope.nonce === 'bigint'
            ? { nonce: Hex.fromNumber(envelope.nonce) }
            : {}),
        ...(typeof envelope.value === 'bigint'
            ? { value: Hex.fromNumber(envelope.value) }
            : {}),
        ...(typeof envelope.maxFeePerBlobGas === 'bigint'
            ? { maxFeePerBlobGas: Hex.fromNumber(envelope.maxFeePerBlobGas) }
            : {}),
        ...(typeof envelope.maxFeePerGas === 'bigint'
            ? { maxFeePerGas: Hex.fromNumber(envelope.maxFeePerGas) }
            : {}),
        ...(typeof envelope.maxPriorityFeePerGas === 'bigint'
            ? { maxPriorityFeePerGas: Hex.fromNumber(envelope.maxPriorityFeePerGas) }
            : {}),
        type: '0x3',
        ...(signature ? Signature.toRpc(signature) : {}),
    };
}
function validate(envelope) {
    try {
        assert(envelope);
        return true;
    }
    catch {
        return false;
    }
}
//# sourceMappingURL=TransactionEnvelopeEip4844.js.map