"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.type = exports.serializedType = void 0;
exports.assert = assert;
exports.deserialize = deserialize;
exports.from = from;
exports.getSignPayload = getSignPayload;
exports.hash = hash;
exports.serialize = serialize;
exports.validate = validate;
const AccessList = require("./AccessList.js");
const Address = require("./Address.js");
const Authorization = require("./Authorization.js");
const Hash = require("./Hash.js");
const Hex = require("./Hex.js");
const Rlp = require("./Rlp.js");
const Signature = require("./Signature.js");
const TransactionEnvelope = require("./TransactionEnvelope.js");
const TransactionEnvelopeEip1559 = require("./TransactionEnvelopeEip1559.js");
exports.serializedType = '0x04';
exports.type = 'eip7702';
function assert(envelope) {
    const { authorizationList } = envelope;
    if (authorizationList) {
        for (const authorization of authorizationList) {
            const { address, chainId } = authorization;
            if (address)
                Address.assert(address, { strict: false });
            if (Number(chainId) < 0)
                throw new TransactionEnvelope.InvalidChainIdError({ chainId });
        }
    }
    TransactionEnvelopeEip1559.assert(envelope);
}
function deserialize(serialized) {
    const transactionArray = Rlp.toHex(Hex.slice(serialized, 1));
    const [chainId, nonce, maxPriorityFeePerGas, maxFeePerGas, gas, to, value, data, accessList, authorizationList, yParity, r, s,] = transactionArray;
    if (!(transactionArray.length === 10 || transactionArray.length === 13))
        throw new TransactionEnvelope.InvalidSerializedError({
            attributes: {
                chainId,
                nonce,
                maxPriorityFeePerGas,
                maxFeePerGas,
                gas,
                to,
                value,
                data,
                accessList,
                authorizationList,
                ...(transactionArray.length > 9
                    ? {
                        yParity,
                        r,
                        s,
                    }
                    : {}),
            },
            serialized,
            type: exports.type,
        });
    let transaction = {
        chainId: Number(chainId),
        type: exports.type,
    };
    if (Hex.validate(to) && to !== '0x')
        transaction.to = to;
    if (Hex.validate(gas) && gas !== '0x')
        transaction.gas = BigInt(gas);
    if (Hex.validate(data) && data !== '0x')
        transaction.data = data;
    if (Hex.validate(nonce))
        transaction.nonce = nonce === '0x' ? 0n : BigInt(nonce);
    if (Hex.validate(value) && value !== '0x')
        transaction.value = BigInt(value);
    if (Hex.validate(maxFeePerGas) && maxFeePerGas !== '0x')
        transaction.maxFeePerGas = BigInt(maxFeePerGas);
    if (Hex.validate(maxPriorityFeePerGas) && maxPriorityFeePerGas !== '0x')
        transaction.maxPriorityFeePerGas = BigInt(maxPriorityFeePerGas);
    if (accessList.length !== 0 && accessList !== '0x')
        transaction.accessList = AccessList.fromTupleList(accessList);
    if (authorizationList !== '0x')
        transaction.authorizationList = Authorization.fromTupleList(authorizationList);
    const signature = r && s && yParity ? Signature.fromTuple([yParity, r, s]) : undefined;
    if (signature)
        transaction = {
            ...transaction,
            ...signature,
        };
    assert(transaction);
    return transaction;
}
function from(envelope, options = {}) {
    const { signature } = options;
    const envelope_ = (typeof envelope === 'string' ? deserialize(envelope) : envelope);
    assert(envelope_);
    return {
        ...envelope_,
        ...(signature ? Signature.from(signature) : {}),
        type: 'eip7702',
    };
}
function getSignPayload(envelope) {
    return hash(envelope, { presign: true });
}
function hash(envelope, options = {}) {
    const { presign } = options;
    return Hash.keccak256(serialize({
        ...envelope,
        ...(presign
            ? {
                r: undefined,
                s: undefined,
                yParity: undefined,
            }
            : {}),
    }));
}
function serialize(envelope, options = {}) {
    const { authorizationList, chainId, gas, nonce, to, value, maxFeePerGas, maxPriorityFeePerGas, accessList, data, input, } = envelope;
    assert(envelope);
    const accessTupleList = AccessList.toTupleList(accessList);
    const authorizationTupleList = Authorization.toTupleList(authorizationList);
    const signature = Signature.extract(options.signature || envelope);
    const serialized = [
        Hex.fromNumber(chainId),
        nonce ? Hex.fromNumber(nonce) : '0x',
        maxPriorityFeePerGas ? Hex.fromNumber(maxPriorityFeePerGas) : '0x',
        maxFeePerGas ? Hex.fromNumber(maxFeePerGas) : '0x',
        gas ? Hex.fromNumber(gas) : '0x',
        to ?? '0x',
        value ? Hex.fromNumber(value) : '0x',
        data ?? input ?? '0x',
        accessTupleList,
        authorizationTupleList,
        ...(signature ? Signature.toTuple(signature) : []),
    ];
    return Hex.concat(exports.serializedType, Rlp.fromHex(serialized));
}
function validate(envelope) {
    try {
        assert(envelope);
        return true;
    }
    catch {
        return false;
    }
}
//# sourceMappingURL=TransactionEnvelopeEip7702.js.map