import { bls12_381 as bls } from '@noble/curves/bls12-381';
import * as Hex from './Hex.js';
/**
 * Converts a BLS point to {@link ox#Bytes.Bytes}.
 *
 * @example
 * ### Public Key to Bytes
 * ```ts twoslash
 * import { Bls, BlsPoint } from 'ox'
 *
 * const publicKey = Bls.getPublicKey({ privateKey: '0x...' })
 * const publicKeyBytes = BlsPoint.toBytes(publicKey)
 * // @log: Uint8Array [172, 175, 255, ...]
 * ```
 *
 * @example
 * ### Signature to Bytes
 * ```ts twoslash
 * import { Bls, BlsPoint } from 'ox'
 *
 * const signature = Bls.sign({ payload: '0x...', privateKey: '0x...' })
 * const signatureBytes = BlsPoint.toBytes(signature)
 * // @log: Uint8Array [172, 175, 255, ...]
 * ```
 *
 * @param point - The BLS point to convert.
 * @returns The bytes representation of the BLS point.
 */
export function toBytes(point) {
    const group = typeof point.z === 'bigint' ? bls.G1 : bls.G2;
    return new group.ProjectivePoint(point.x, point.y, point.z).toRawBytes();
}
// eslint-disable-next-line jsdoc/require-jsdoc
export function toHex(point) {
    return Hex.fromBytes(toBytes(point));
}
// eslint-disable-next-line jsdoc/require-jsdoc
export function fromBytes(bytes) {
    const group = bytes.length === 48 ? bls.G1 : bls.G2;
    const point = group.ProjectivePoint.fromHex(bytes);
    return {
        x: point.px,
        y: point.py,
        z: point.pz,
    };
}
// eslint-disable-next-line jsdoc/require-jsdoc
export function fromHex(hex, group) {
    return fromBytes(Hex.toBytes(hex), group);
}
//# sourceMappingURL=BlsPoint.js.map