import * as Hex from './Hex.js';
/**
 * Converts a {@link ox#Filter.Rpc} to an {@link ox#Filter.Filter}.
 *
 * @example
 * ```ts twoslash
 * import { Filter } from 'ox'
 *
 * const filter = Filter.fromRpc({
 *   address: '0xd3cda913deb6f67967b99d671a681250403edf27',
 *   fromBlock: 'latest',
 *   toBlock: '0x010f2c',
 *   topics: [
 *     '0xddf252ad1be2c89b69c2b068fc378daa952ba7f163c4a11628f55a4df523b3ef',
 *     null,
 *     '0x0000000000000000000000000c04d9e9278ec5e4d424476d3ebec70cb5d648d1',
 *   ],
 * })
 * // @log: {
 * // @log:   address: '0xd3cda913deb6f67967b99d671a681250403edf27',
 * // @log:   fromBlock: 'latest',
 * // @log:   toBlock: 69420n,
 * // @log:   topics: [
 * // @log:     '0xddf252ad1be2c89b69c2b068fc378daa952ba7f163c4a11628f55a4df523b3ef',
 * // @log:     null,
 * // @log:     '0x0000000000000000000000000c04d9e9278ec5e4d424476d3ebec70cb5d648d1',
 * // @log:   ],
 * // @log: }
 * ```
 *
 * @param filter - The RPC filter to convert.
 * @returns An instantiated {@link ox#Filter.Filter}.
 */
export function fromRpc(filter) {
    const { fromBlock, toBlock } = filter;
    return {
        ...filter,
        ...(fromBlock && {
            fromBlock: Hex.validate(fromBlock, { strict: false })
                ? BigInt(fromBlock)
                : fromBlock,
        }),
        ...(toBlock && {
            toBlock: Hex.validate(toBlock, { strict: false })
                ? BigInt(toBlock)
                : toBlock,
        }),
    };
}
/**
 * Converts a {@link ox#Filter.Filter} to a {@link ox#Filter.Rpc}.
 *
 * @example
 * ```ts twoslash
 * import { AbiEvent, Filter } from 'ox'
 *
 * const transfer = AbiEvent.from('event Transfer(address indexed, address indexed, uint256)')
 * const { topics } = AbiEvent.encode(transfer)
 *
 * const filter = Filter.toRpc({
 *   address: '0xfba3912ca04dd458c843e2ee08967fc04f3579c2',
 *   topics,
 * })
 * // @log: {
 * // @log:   address: '0xfba3912ca04dd458c843e2ee08967fc04f3579c2',
 * // @log:   topics: [
 * // @log:     '0xddf252ad1be2c89b69c2b068fc378daa952ba7f163c4a11628f55a4df523b3ef',
 * // @log:   ],
 * // @log: }
 * ```
 *
 * @param filter - The filter to convert.
 * @returns An RPC filter.
 */
export function toRpc(filter) {
    const { address, topics, fromBlock, toBlock } = filter;
    return {
        ...(address && { address }),
        ...(topics && { topics }),
        ...(typeof fromBlock !== 'undefined'
            ? {
                fromBlock: typeof fromBlock === 'bigint'
                    ? Hex.fromNumber(fromBlock)
                    : fromBlock,
            }
            : {}),
        ...(typeof toBlock !== 'undefined'
            ? {
                toBlock: typeof toBlock === 'bigint' ? Hex.fromNumber(toBlock) : toBlock,
            }
            : {}),
    };
}
//# sourceMappingURL=Filter.js.map