import { generateMnemonic, mnemonicToSeedSync, validateMnemonic, } from '@scure/bip39';
import * as Bytes from './Bytes.js';
import * as HdKey from './HdKey.js';
export { path } from './HdKey.js';
export { czech, english, french, italian, japanese, korean, portuguese, simplifiedChinese, spanish, traditionalChinese, } from './internal/mnemonic/wordlists.js';
/**
 * Generates a random mnemonic.
 *
 * @example
 * ```ts twoslash
 * import { Mnemonic } from 'ox'
 *
 * const mnemonic = Mnemonic.random(Mnemonic.english)
 * // @log: 'buyer zoo end danger ice capable shrug naive twist relief mass bonus'
 * ```
 *
 * @param wordlist - The wordlist to use.
 * @param options - Generation options.
 * @returns The mnemonic.
 */
export function random(wordlist, options = {}) {
    const { strength = 128 } = options;
    return generateMnemonic(wordlist, strength);
}
/**
 * Converts a mnemonic to a HD Key.
 *
 * @example
 * ```ts twoslash
 * import { Mnemonic } from 'ox'
 *
 * const mnemonic = Mnemonic.random(Mnemonic.english)
 * const hdKey = Mnemonic.toHdKey(mnemonic)
 * ```
 *
 * @example
 * ### Path Derivation
 *
 * You can derive a HD Key at a specific path using `derive`:
 *
 * ```ts twoslash
 * import { Mnemonic } from 'ox'
 *
 * const mnemonic = Mnemonic.random(Mnemonic.english)
 * const hdKey = Mnemonic.toHdKey(mnemonic).derive(Mnemonic.path({ index: 1 }))
 * ```
 *
 * @param mnemonic - The mnemonic to convert.
 * @param options - Conversion options.
 * @returns The HD Key.
 */
export function toHdKey(mnemonic, options = {}) {
    const { passphrase } = options;
    const seed = toSeed(mnemonic, { passphrase });
    return HdKey.fromSeed(seed);
}
/**
 * Converts a mnemonic to a private key.
 *
 * @example
 * ```ts twoslash
 * import { Mnemonic } from 'ox'
 *
 * const mnemonic = Mnemonic.random(Mnemonic.english)
 * const privateKey = Mnemonic.toPrivateKey(mnemonic)
 * // @log: '0x...'
 * ```
 *
 * @example
 * ### Paths
 *
 * You can derive a private key at a specific path using the `path` option.
 *
 * ```ts twoslash
 * import { Mnemonic } from 'ox'
 *
 * const mnemonic = Mnemonic.random(Mnemonic.english)
 * const privateKey = Mnemonic.toPrivateKey(mnemonic, {
 *   path: Mnemonic.path({ index: 1 }) // 'm/44'/60'/0'/0/1' // [!code focus]
 * })
 * // @log: '0x...'
 * ```
 *
 * @param mnemonic - The mnemonic to convert.
 * @param options - Conversion options.
 * @returns The private key.
 */
export function toPrivateKey(mnemonic, options = {}) {
    const { path = HdKey.path(), passphrase } = options;
    const hdKey = toHdKey(mnemonic, { passphrase }).derive(path);
    if (options.as === 'Bytes')
        return Bytes.from(hdKey.privateKey);
    return hdKey.privateKey;
}
/**
 * Converts a mnemonic to a master seed.
 *
 * @example
 * ```ts twoslash
 * import { Mnemonic } from 'ox'
 *
 * const mnemonic = Mnemonic.random(Mnemonic.english)
 * const seed = Mnemonic.toSeed(mnemonic)
 * // @log: Uint8Array [...64 bytes]
 * ```
 *
 * @param mnemonic - The mnemonic to convert.
 * @param options - Conversion options.
 * @returns The master seed.
 */
export function toSeed(mnemonic, options = {}) {
    const { passphrase } = options;
    const seed = mnemonicToSeedSync(mnemonic, passphrase);
    if (options.as === 'Hex')
        return Bytes.toHex(seed);
    return seed;
}
/**
 * Checks if a mnemonic is valid, given a wordlist.
 *
 * @example
 * ```ts twoslash
 * import { Mnemonic } from 'ox'
 *
 * const mnemonic = Mnemonic.validate(
 *   'buyer zoo end danger ice capable shrug naive twist relief mass bonus',
 *   Mnemonic.english
 * )
 * // @log: true
 * ```
 *
 * @param mnemonic - The mnemonic to validate.
 * @param wordlist - The wordlist to use.
 * @returns Whether the mnemonic is valid.
 */
export function validate(mnemonic, wordlist) {
    return validateMnemonic(mnemonic, wordlist);
}
//# sourceMappingURL=Mnemonic.js.map