/**
 * Utility functions for working with [ERC-8010 wrapped signatures](https://eips.ethereum.org/EIPS/eip-8010#specification).
 *
 * @example
 * ```ts twoslash
 * import { Authorization, Secp256k1, PersonalMessage } from 'ox'
 * import { SignatureErc8010 } from 'ox/erc8010' // [!code focus]
 *
 * const authorization = Authorization.from({
 *   address: '0x1234567890abcdef1234567890abcdef12345678',
 *   chainId: 1,
 *   nonce: 69n,
 * })
 *
 * const authorizationSignature = Secp256k1.sign({
 *   payload: Authorization.getSignPayload(authorization),
 *   privateKey: '0x...',
 * })
 *
 * const authorizationSigned = Authorization.from(authorization, {
 *   signature: authorizationSignature,
 * })
 *
 * const signature = Secp256k1.sign({
 *   payload: PersonalMessage.getSignPayload('0xdeadbeef'),
 *   privateKey: '0x...',
 * })
 *
 * const wrapped = SignatureErc8010.wrap({ // [!code focus]
 *   authorization: authorizationSigned, // [!code focus]
 *   data: '0xcafebabe', // [!code focus]
 *   signature, // [!code focus]
 * }) // [!code focus]
 * // @log: '0x000000000000000000000000cafebabecafebabecafebabecafebabecafebabe000000000000000000000000000000000000000000000000000000000000006000000000000000000000000000000000000000000000000000000000000000a00000000000000000000000000000000000000000000000000000000000000004deadbeef000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000041fa78c5905fb0b9d6066ef531f962a62bc6ef0d5eb59ecb134056d206f75aaed7780926ff2601a935c2c79707d9e1799948c9f19dcdde1e090e903b19a07923d01c000000000000000000000000000000000000000000000000000000000000008010801080108010801080108010801080108010801080108010801080108010'
 * ```
 *
 * @category ERC-8010
 */
export * as SignatureErc8010 from './SignatureErc8010.js';
//# sourceMappingURL=index.js.map