import * as Bytes from './Bytes.js';
import * as Errors from './Errors.js';
import * as Hex from './Hex.js';
import * as Cursor from './internal/cursor.js';
import type { ExactPartial, RecursiveArray } from './internal/types.js';
/**
 * Decodes a Recursive-Length Prefix (RLP) value into a {@link ox#Bytes.Bytes} value.
 *
 * @example
 * ```ts twoslash
 * import { Rlp } from 'ox'
 * Rlp.toBytes('0x8b68656c6c6f20776f726c64')
 * // Uint8Array([139, 104, 101, 108, 108, 111,  32, 119, 111, 114, 108, 100])
 * ```
 *
 * @param value - The value to decode.
 * @returns The decoded {@link ox#Bytes.Bytes} value.
 */
export declare function toBytes(value: Bytes.Bytes | Hex.Hex): RecursiveArray<Bytes.Bytes>;
export declare namespace toBytes {
    type ErrorType = to.ErrorType;
}
/**
 * Decodes a Recursive-Length Prefix (RLP) value into a {@link ox#Hex.Hex} value.
 *
 * @example
 * ```ts twoslash
 * import { Rlp } from 'ox'
 * Rlp.toHex('0x8b68656c6c6f20776f726c64')
 * // 0x68656c6c6f20776f726c64
 * ```
 *
 * @param value - The value to decode.
 * @returns The decoded {@link ox#Hex.Hex} value.
 */
export declare function toHex(value: Bytes.Bytes | Hex.Hex): RecursiveArray<Hex.Hex>;
export declare namespace toHex {
    type ErrorType = to.ErrorType;
}
/** @internal */
export declare function to<value extends Bytes.Bytes | Hex.Hex, to extends 'Hex' | 'Bytes'>(value: value, to: to | 'Hex' | 'Bytes'): to.ReturnType<to>;
/** @internal */
export declare namespace to {
    type ReturnType<to extends 'Hex' | 'Bytes' = 'Hex' | 'Bytes'> = (to extends 'Bytes' ? RecursiveArray<Bytes.Bytes> : never) | (to extends 'Hex' ? RecursiveArray<Hex.Hex> : never);
    type ErrorType = Bytes.fromHex.ErrorType | decodeRlpCursor.ErrorType | Cursor.create.ErrorType | Hex.InvalidLengthError | Errors.GlobalErrorType;
}
/** @internal */
/** @internal */
export declare function decodeRlpCursor<to extends 'Hex' | 'Bytes' = 'Hex'>(cursor: Cursor.Cursor, to?: to | 'Hex' | 'Bytes' | undefined): decodeRlpCursor.ReturnType<to>;
/** @internal */
export declare namespace decodeRlpCursor {
    type ReturnType<to extends 'Hex' | 'Bytes' = 'Hex'> = to.ReturnType<to>;
    type ErrorType = Hex.fromBytes.ErrorType | readLength.ErrorType | readList.ErrorType | Errors.GlobalErrorType;
}
/** @internal */
export declare function readLength(cursor: Cursor.Cursor, prefix: number, offset: number): number;
/** @internal */
export declare namespace readLength {
    type ErrorType = Errors.BaseError | Errors.GlobalErrorType;
}
/** @internal */
export declare function readList<to extends 'Hex' | 'Bytes'>(cursor: Cursor.Cursor, length: number, to: to | 'Hex' | 'Bytes'): decodeRlpCursor.ReturnType<to>[];
/** @internal */
export declare namespace readList {
    type ErrorType = Errors.GlobalErrorType;
}
/**
 * Encodes a {@link ox#Bytes.Bytes} or {@link ox#Hex.Hex} value into a Recursive-Length Prefix (RLP) value.
 *
 * @example
 * ```ts twoslash
 * import { Bytes, Rlp } from 'ox'
 *
 * Rlp.from('0x68656c6c6f20776f726c64', { as: 'Hex' })
 * // @log: 0x8b68656c6c6f20776f726c64
 *
 * Rlp.from(Bytes.from([139, 104, 101, 108, 108, 111,  32, 119, 111, 114, 108, 100]), { as: 'Bytes' })
 * // @log: Uint8Array([104, 101, 108, 108, 111,  32, 119, 111, 114, 108, 100])
 * ```
 *
 * @param value - The {@link ox#Bytes.Bytes} or {@link ox#Hex.Hex} value to encode.
 * @param options - Options.
 * @returns The RLP value.
 */
export declare function from<as extends 'Hex' | 'Bytes'>(value: RecursiveArray<Bytes.Bytes> | RecursiveArray<Hex.Hex>, options: from.Options<as>): from.ReturnType<as>;
export declare namespace from {
    type Options<as extends 'Hex' | 'Bytes'> = {
        /** The type to convert the RLP value to. */
        as: as | 'Hex' | 'Bytes';
    };
    type ReturnType<as extends 'Hex' | 'Bytes'> = (as extends 'Bytes' ? Bytes.Bytes : never) | (as extends 'Hex' ? Hex.Hex : never);
    type ErrorType = Cursor.create.ErrorType | Hex.fromBytes.ErrorType | Bytes.fromHex.ErrorType | Errors.GlobalErrorType;
}
/**
 * Encodes a {@link ox#Bytes.Bytes} value into a Recursive-Length Prefix (RLP) value.
 *
 * @example
 * ```ts twoslash
 * import { Bytes, Rlp } from 'ox'
 *
 * Rlp.fromBytes(Bytes.from([139, 104, 101, 108, 108, 111,  32, 119, 111, 114, 108, 100]))
 * // @log: Uint8Array([104, 101, 108, 108, 111,  32, 119, 111, 114, 108, 100])
 * ```
 *
 * @param bytes - The {@link ox#Bytes.Bytes} value to encode.
 * @param options - Options.
 * @returns The RLP value.
 */
export declare function fromBytes<as extends 'Hex' | 'Bytes' = 'Bytes'>(bytes: RecursiveArray<Bytes.Bytes>, options?: fromBytes.Options<as>): fromBytes.ReturnType<as>;
export declare namespace fromBytes {
    type Options<as extends 'Hex' | 'Bytes' = 'Bytes'> = ExactPartial<from.Options<as>>;
    type ReturnType<as extends 'Hex' | 'Bytes' = 'Bytes'> = from.ReturnType<as>;
    type ErrorType = from.ErrorType | Errors.GlobalErrorType;
}
/**
 * Encodes a {@link ox#Hex.Hex} value into a Recursive-Length Prefix (RLP) value.
 *
 * @example
 * ```ts twoslash
 * import { Rlp } from 'ox'
 *
 * Rlp.fromHex('0x68656c6c6f20776f726c64')
 * // @log: 0x8b68656c6c6f20776f726c64
 * ```
 *
 * @param hex - The {@link ox#Hex.Hex} value to encode.
 * @param options - Options.
 * @returns The RLP value.
 */
export declare function fromHex<as extends 'Hex' | 'Bytes' = 'Hex'>(hex: RecursiveArray<Hex.Hex>, options?: fromHex.Options<as>): fromHex.ReturnType<as>;
export declare namespace fromHex {
    type Options<as extends 'Hex' | 'Bytes' = 'Hex'> = ExactPartial<from.Options<as>>;
    type ReturnType<as extends 'Hex' | 'Bytes' = 'Hex'> = from.ReturnType<as>;
    type ErrorType = from.ErrorType | Errors.GlobalErrorType;
}
//# sourceMappingURL=Rlp.d.ts.map