import * as Address from './Address.js';
import * as Errors from './Errors.js';
import type { ExactPartial } from './internal/types.js';
export declare const domainRegex: RegExp;
export declare const ipRegex: RegExp;
export declare const localhostRegex: RegExp;
export declare const nonceRegex: RegExp;
export declare const schemeRegex: RegExp;
export declare const prefixRegex: RegExp;
export declare const suffixRegex: RegExp;
/** [EIP-4361](https://eips.ethereum.org/EIPS/eip-4361) message fields. */
export type Message = {
    /**
     * The Ethereum address performing the signing.
     */
    address: Address.Address;
    /**
     * The [EIP-155](https://eips.ethereum.org/EIPS/eip-155) Chain ID to which the session is bound,
     */
    chainId: number;
    /**
     * [RFC 3986](https://www.rfc-editor.org/rfc/rfc3986) authority that is requesting the signing.
     */
    domain: string;
    /**
     * Time when the signed authentication message is no longer valid.
     */
    expirationTime?: Date | undefined;
    /**
     * Time when the message was generated, typically the current time.
     */
    issuedAt?: Date | undefined;
    /**
     * A random string typically chosen by the relying party and used to prevent replay attacks.
     */
    nonce: string;
    /**
     * Time when the signed authentication message will become valid.
     */
    notBefore?: Date | undefined;
    /**
     * A system-specific identifier that may be used to uniquely refer to the sign-in request.
     */
    requestId?: string | undefined;
    /**
     * A list of information or references to information the user wishes to have resolved as part of authentication by the relying party.
     */
    resources?: string[] | undefined;
    /**
     * [RFC 3986](https://www.rfc-editor.org/rfc/rfc3986#section-3.1) URI scheme of the origin of the request.
     */
    scheme?: string | undefined;
    /**
     * A human-readable ASCII assertion that the user will sign.
     */
    statement?: string | undefined;
    /**
     * [RFC 3986](https://www.rfc-editor.org/rfc/rfc3986) URI referring to the resource that is the subject of the signing (as in the subject of a claim).
     */
    uri: string;
    /**
     * The current version of the SIWE Message.
     */
    version: '1';
};
/**
 * Creates [EIP-4361](https://eips.ethereum.org/EIPS/eip-4361) formatted message.
 *
 * @example
 * ```ts twoslash
 * import { Siwe } from 'ox'
 *
 * Siwe.createMessage({
 *   address: '0xA0Cf798816D4b9b9866b5330EEa46a18382f251e',
 *   chainId: 1,
 *   domain: 'example.com',
 *   nonce: 'foobarbaz',
 *   uri: 'https://example.com/path',
 *   version: '1',
 * })
 * // @log: "example.com wants you to sign in with your Ethereum account:
 * // @log: 0xA0Cf798816D4b9b9866b5330EEa46a18382f251e
 * // @log:
 * // @log:
 * // @log: URI: https://example.com/path
 * // @log: Version: 1
 * // @log: Chain ID: 1
 * // @log: Nonce: foobarbaz
 * // @log: Issued At: 2023-02-01T00:00:00.000Z"
 * ```
 *
 * @param value - Values to use when creating EIP-4361 formatted message.
 * @returns EIP-4361 formatted message.
 */
export declare function createMessage(value: Message): string;
export declare namespace createMessage {
    type ErrorType = Address.from.ErrorType | InvalidMessageFieldError | Errors.GlobalErrorType;
}
/**
 * Generates random [EIP-4361](https://eips.ethereum.org/EIPS/eip-4361) nonce.
 *
 * @example
 * ```ts twoslash
 * import { Siwe } from 'ox'
 *
 * Siwe.generateNonce()
 * // @log: '65ed4681d4efe0270b923ff5f4b097b1c95974dc33aeebecd5724c42fd86dfd25dc70b27ef836b2aa22e68f19ebcccc1'
 * ```
 *
 * @returns Random nonce.
 */
export declare function generateNonce(): string;
/**
 * Check if the given URI is a valid [RFC 3986](https://www.rfc-editor.org/rfc/rfc3986) URI.
 *
 * @example
 * ```ts twoslash
 * import { Siwe } from 'ox'
 *
 * Siwe.isUri('https://example.com/foo')
 * // @log: true
 * ```
 *
 * @param value - Value to check.
 * @returns `false` if invalid, otherwise the valid URI.
 */
export declare function isUri(value: string): false | string;
/**
 * [EIP-4361](https://eips.ethereum.org/EIPS/eip-4361) formatted message into message fields object.
 *
 * @example
 * ```ts twoslash
 * import { Siwe } from 'ox'
 *
 * Siwe.parseMessage(`example.com wants you to sign in with your Ethereum account:
 * 0xA0Cf798816D4b9b9866b5330EEa46a18382f251e
 *
 * I accept the ExampleOrg Terms of Service: https://example.com/tos
 *
 * URI: https://example.com/path
 * Version: 1
 * Chain ID: 1
 * Nonce: foobarbaz
 * Issued At: 2023-02-01T00:00:00.000Z`)
 * // @log: {
 * // @log:   address: '0xA0Cf798816D4b9b9866b5330EEa46a18382f251e',
 * // @log:   chainId: 1,
 * // @log:   domain: 'example.com',
 * // @log:   issuedAt: '2023-02-01T00:00:00.000Z',
 * // @log:   nonce: 'foobarbaz',
 * // @log:   statement: 'I accept the ExampleOrg Terms of Service: https://example.com/tos',
 * // @log:   uri: 'https://example.com/path',
 * // @log:   version: '1',
 * // @log: }
 * ```
 *
 * @param message - [EIP-4361](https://eips.ethereum.org/EIPS/eip-4361) formatted message.
 * @returns Message fields object.
 */
export declare function parseMessage(message: string): ExactPartial<Message>;
/**
 * Validates [EIP-4361](https://eips.ethereum.org/EIPS/eip-4361) message.
 *
 * @example
 * ```ts twoslash
 * import { Siwe } from 'ox'
 *
 * Siwe.validateMessage({
 *   address: '0xA0Cf798816D4b9b9866b5330EEa46a18382f251e',
 *   domain: 'example.com',
 *   message: {
 *     address: '0xA0Cf798816D4b9b9866b5330EEa46a18382f251e',
 *     chainId: 1,
 *     domain: 'example.com',
 *     nonce: 'foobarbaz',
 *     uri: 'https://example.com/path',
 *     version: '1',
 *   },
 *   nonce: 'foobarbaz',
 * })
 * // @log: true
 * ```
 *
 * @param value - Values to use when validating EIP-4361 formatted message.
 * @returns Whether the message is valid.
 */
export declare function validateMessage(value: validateMessage.Value): boolean;
export declare namespace validateMessage {
    interface Value {
        /**
         * Ethereum address to check against.
         */
        address?: Address.Address | undefined;
        /**
         * [RFC 3986](https://www.rfc-editor.org/rfc/rfc3986) authority to check against.
         */
        domain?: string | undefined;
        /**
         * EIP-4361 message fields.
         */
        message: ExactPartial<Message>;
        /**
         * Random string to check against.
         */
        nonce?: string | undefined;
        /**
         * [RFC 3986](https://www.rfc-editor.org/rfc/rfc3986#section-3.1) URI scheme to check against.
         */
        scheme?: string | undefined;
        /**
         * Current time to check optional `expirationTime` and `notBefore` fields.
         *
         * @default new Date()
         */
        time?: Date | undefined;
    }
}
/**
 * Thrown when a field in a SIWE Message is invalid.
 *
 * @example
 * ```ts twoslash
 * import { Siwe } from 'ox'
 *
 * Siwe.createMessage({
 *   address: '0xA0Cf798816D4b9b9866b5330EEa46a18382f251e',
 *   chainId: 1.1,
 *   domain: 'example.com',
 *   nonce: 'foobarbaz',
 *   uri: 'https://example.com/path',
 *   version: '1',
 * })
 * // @error: Siwe.InvalidMessageFieldError: Invalid Sign-In with Ethereum message field "chainId".
 * // @error: - Chain ID must be a EIP-155 chain ID.
 * // @error: - See https://eips.ethereum.org/EIPS/eip-155
 * // @error: Provided value: 1.1
 * ```
 */
export declare class InvalidMessageFieldError extends Errors.BaseError {
    readonly name = "Siwe.InvalidMessageFieldError";
    constructor(parameters: {
        field: string;
        metaMessages?: string[] | undefined;
    });
}
//# sourceMappingURL=Siwe.d.ts.map