import type * as Errors from './Errors.js';
import * as Hex from './Hex.js';
/** A Withdrawal as defined in the [Execution API specification](https://github.com/ethereum/execution-apis/blob/main/src/schemas/withdrawal.yaml). */
export type Withdrawal<bigintType = bigint, numberType = number> = {
    address: Hex.Hex;
    amount: bigintType;
    index: numberType;
    validatorIndex: numberType;
};
/** An RPC Withdrawal as defined in the [Execution API specification](https://github.com/ethereum/execution-apis/blob/main/src/schemas/withdrawal.yaml). */
export type Rpc = Withdrawal<Hex.Hex, Hex.Hex>;
/**
 * Converts a {@link ox#Withdrawal.Rpc} to an {@link ox#Withdrawal.Withdrawal}.
 *
 * @example
 * ```ts twoslash
 * import { Withdrawal } from 'ox'
 *
 * const withdrawal = Withdrawal.fromRpc({
 *   address: '0x00000000219ab540356cBB839Cbe05303d7705Fa',
 *   amount: '0x620323',
 *   index: '0x0',
 *   validatorIndex: '0x1',
 * })
 * // @log: {
 * // @log:   address: '0x00000000219ab540356cBB839Cbe05303d7705Fa',
 * // @log:   amount: 6423331n,
 * // @log:   index: 0,
 * // @log:   validatorIndex: 1
 * // @log: }
 * ```
 *
 * @param withdrawal - The RPC withdrawal to convert.
 * @returns An instantiated {@link ox#Withdrawal.Withdrawal}.
 */
export declare function fromRpc(withdrawal: Rpc): Withdrawal;
export declare namespace fromRpc {
    type ErrorType = Errors.GlobalErrorType;
}
/**
 * Converts a {@link ox#Withdrawal.Withdrawal} to an {@link ox#Withdrawal.Rpc}.
 *
 * @example
 * ```ts twoslash
 * import { Withdrawal } from 'ox'
 *
 * const withdrawal = Withdrawal.toRpc({
 *   address: '0x00000000219ab540356cBB839Cbe05303d7705Fa',
 *   amount: 6423331n,
 *   index: 0,
 *   validatorIndex: 1,
 * })
 * // @log: {
 * // @log:   address: '0x00000000219ab540356cBB839Cbe05303d7705Fa',
 * // @log:   amount: '0x620323',
 * // @log:   index: '0x0',
 * // @log:   validatorIndex: '0x1',
 * // @log: }
 * ```
 *
 * @param withdrawal - The Withdrawal to convert.
 * @returns An RPC Withdrawal.
 */
export declare function toRpc(withdrawal: Withdrawal): Rpc;
export declare namespace toRpc {
    type ErrorType = Errors.GlobalErrorType;
}
//# sourceMappingURL=Withdrawal.d.ts.map