import type * as Address from '../core/Address.js';
import * as Authorization from '../core/Authorization.js';
import * as Errors from '../core/Errors.js';
import * as Hex from '../core/Hex.js';
/** Unwrapped ERC-8010 signature. */
export type Unwrapped = {
    /** Authorization signed by the delegatee. */
    authorization: Authorization.Authorization<true>;
    /** Data to initialize the delegation. */
    data?: Hex.Hex | undefined;
    /** The original signature. */
    signature: Hex.Hex;
    /** Address of the initializer. */
    to?: Address.Address | undefined;
};
/** Wrapped ERC-8010 signature. */
export type Wrapped = Hex.Hex;
/**
 * Magic bytes used to identify ERC-8010 wrapped signatures.
 */
export declare const magicBytes: "0x8010801080108010801080108010801080108010801080108010801080108010";
/** Suffix ABI parameters for the ERC-8010 wrapped signature. */
export declare const suffixParameters: readonly [{
    readonly type: "tuple";
    readonly components: readonly [{
        readonly type: "uint256";
        readonly name: "chainId";
    }, {
        readonly type: "address";
        readonly name: "delegation";
    }, {
        readonly type: "uint256";
        readonly name: "nonce";
    }, {
        readonly type: "uint8";
        readonly name: "yParity";
    }, {
        readonly type: "uint256";
        readonly name: "r";
    }, {
        readonly type: "uint256";
        readonly name: "s";
    }];
}, {
    readonly type: "address";
    readonly name: "to";
}, {
    readonly type: "bytes";
    readonly name: "data";
}];
/**
 * Asserts that the wrapped signature is valid.
 *
 * @example
 * ```ts twoslash
 * import { SignatureErc8010 } from 'ox/erc8010'
 *
 * SignatureErc8010.assert('0xdeadbeef')
 * // @error: InvalidWrappedSignatureError: Value `0xdeadbeef` is an invalid ERC-8010 wrapped signature.
 * ```
 *
 * @param value - The value to assert.
 */
export declare function assert(value: Unwrapped | Wrapped): void;
export declare namespace assert {
    type ErrorType = InvalidWrappedSignatureError | Hex.slice.ErrorType | Errors.GlobalErrorType;
}
/**
 * Parses an [ERC-8010 wrapped signature](https://github.com/jxom/ERCs/blob/16f7e3891fff2e1e9c25dea0485497739db8a816/ERCS/erc-8010.md) into its constituent parts.
 *
 * @example
 * ```ts twoslash
 * // @noErrors
 * import { Secp256k1 } from 'ox'
 * import { SignatureErc8010 } from 'ox/erc8010' // [!code focus]
 *
 * const signature = Secp256k1.sign({
 *   payload: '0x...',
 *   privateKey: '0x...',
 * })
 *
 * // Instantiate from serialized format. // [!code focus]
 * const wrapped = SignatureErc8010.from('0x...') // [!code focus]
 * // @log: { authorization: { ... }, data: '0x...', signature: { ... } } // [!code focus]
 *
 * // Instantiate from constituent parts. // [!code focus]
 * const wrapped = SignatureErc8010.from({ // [!code focus]
 *   authorization: { ... }, // [!code focus]
 *   data: '0x...', // [!code focus]
 *   signature, // [!code focus]
 * })
 * // @log: { authorization: { ... }, data: '0x...', signature: { ... } }
 * ```
 *
 * @param value - Value to parse.
 * @returns Parsed value.
 */
export declare function from(value: Unwrapped | Wrapped): Unwrapped;
export declare namespace from {
    type ErrorType = unwrap.ErrorType | Errors.GlobalErrorType;
}
/**
 * Unwraps an [ERC-8010 wrapped signature](https://github.com/jxom/ERCs/blob/16f7e3891fff2e1e9c25dea0485497739db8a816/ERCS/erc-8010.md) into its constituent parts.
 *
 * @example
 * ```ts twoslash
 * import { SignatureErc8010 } from 'ox/erc8010'
 *
 * const { authorization, data, signature } = SignatureErc8010.unwrap('0x...')
 * ```
 *
 * @param wrapped - Wrapped signature to unwrap.
 * @returns Unwrapped signature.
 */
export declare function unwrap(wrapped: Wrapped): Unwrapped;
export declare namespace unwrap {
    type ErrorType = assert.ErrorType | Errors.GlobalErrorType;
}
/**
 * Wraps a signature into [ERC-8010 format](https://github.com/jxom/ERCs/blob/16f7e3891fff2e1e9c25dea0485497739db8a816/ERCS/erc-8010.md).
 *
 * @example
 * ```ts twoslash
 * // @noErrors
 * import { Secp256k1 } from 'ox'
 * import { SignatureErc8010 } from 'ox/erc8010' // [!code focus]
 *
 * const signature = Secp256k1.sign({
 *   payload: '0x...',
 *   privateKey: '0x...',
 * })
 *
 * const wrapped = SignatureErc8010.wrap({ // [!code focus]
 *   authorization: { ... }, // [!code focus]
 *   data: '0xdeadbeef', // [!code focus]
 *   signature, // [!code focus]
 * }) // [!code focus]
 * ```
 *
 * @param value - Values to wrap.
 * @returns Wrapped signature.
 */
export declare function wrap(value: Unwrapped): Wrapped;
export declare namespace wrap {
    type ErrorType = Errors.GlobalErrorType;
}
/**
 * Validates a wrapped signature. Returns `true` if the wrapped signature is valid, `false` otherwise.
 *
 * @example
 * ```ts twoslash
 * import { SignatureErc8010 } from 'ox/erc8010'
 *
 * const valid = SignatureErc8010.validate('0xdeadbeef')
 * // @log: false
 * ```
 *
 * @param value - The value to validate.
 * @returns `true` if the value is valid, `false` otherwise.
 */
export declare function validate(value: Unwrapped | Wrapped): boolean;
export declare namespace validate {
    type ErrorType = Errors.GlobalErrorType;
}
/** Thrown when the ERC-8010 wrapped signature is invalid. */
export declare class InvalidWrappedSignatureError extends Errors.BaseError {
    readonly name = "SignatureErc8010.InvalidWrappedSignatureError";
    constructor(wrapped: Wrapped);
}
//# sourceMappingURL=SignatureErc8010.d.ts.map